function f = MakeONFilterExt(Type,Par)
%  MakeONFilterExt is an extension of:
%  MakeONFilter -- Generate Orthonormal QMF Filter for Wavelet Transform
%  Usage
%    qmf = MakeONFilterExt(Type,Par)
%  Inputs
%    Type   string, 'Haar', 'Beylkin', 'Coiflet', 'Daubechies',
%           'Symmlet', 'Vaidyanathan','Battle', 'Lemarie','Pollen'
%    Par    integer or angle(s), it is a parameter related to either the support and vanishing
%           moments of the wavelets or the angle, explained below for each wavelet.
%
% Outputs
%    qmf    quadrature mirror filter
%
%  Description
%    The Haar filter (which could be considered a Daubechies-2) was the
%    first wavelet, though not called as such, and is discontinuous. Par is
%    arbitrary.
%
%    The Coiflet filters are designed to give both the mother and father
%    wavelets 2*Par vanishing moments; here Par may be one of 1,2,...,9.
%   
%
%    The Daubechies filters are minimal phase filters that generate wavelets
%    which have a minimal support for a given number of vanishing moments.
%    They are indexed by their length, Par, which may be one of
%    4,6,8, ...,50. The number of vanishing moments is par/2.
%
%    Symmlets are also wavelets within a minimum size support for a given 
%    number of vanishing moments, but they are as symmetrical as possible,
%    as opposed to the Daubechies filters which are highly asymmetrical.
%    They are indexed by Par, which specifies the number of vanishing
%    moments and is equal to half the size of the support. It ranges 
%    from 4 to 25. Symmlet1, 2, and 3 coincide with Haar, Daubechies 4, and
%    6, respectively.
%
%    The Beylkin filter places roots for the frequency response function
%    close to the Nyquist frequency on the real axis.
%
%    The Vaidyanathan filter gives an exact reconstruction, but does not
%    satisfy any moment condition.  The filter has been optimized for
%    speech coding.
%
%    The Battle-Lemarie filter generate spline orthogonal wavelet basis.
%    The parameter Par gives the degree of the spline. The number of 
%    vanishing moments is Par+1.
%    
%    The Lemarie filter is IFR symmetric orthogonal filter curtailed to 50
%    significant coefficients
%
%    The Pollen family is indexed by angle(s). If a single angle from [0, 2*pi] 
%    is supplied, the output is 4 tap filter. Daubechies 4 corresponds to
%    angle pi/6. If Par is two dimensional [phi1, phi2], then the output is
%    6 tap filter.
%
%
%  See Also
%    FWT_PO, IWT_PO, FWT2_PO, IWT2_PO, WPAnalysis
%
%  References
%    The books by Daubechies and Wickerhauser.
%

if strncmpi(Type,'Haar',2),
	f = [1 1];
end

if strncmpi(Type,'Beylkin',2),
	f = [	.099305765374,	.424215360813,	.699825214057	...
			.449718251149,	-.110927598348,	-.264497231446	...
			.026900308804,	.155538731877,	-.017520746267	...
			-.088543630623,	.019679866044,	.042916387274	...
			-.017460408696,	-.014365807969,	.010040411845	...
			.001484234782,	-.002736031626,	.000640485329	];
end

if strncmpi(Type,'Coiflet',3),
	if Par==1,
	f = [   -0.072732619512526448024439375253,  0.337897662457481769674756011043, ...
            0.85257202021160042044972311261,    0.384864846864857747251754521072, ...
            -0.072732619512526448024439375253,  -0.0156557281357919925256661700098  ];
	end
    
    if Par==2,
    f = [   0.016387336463203640427491273259,    -0.041464936786871774009717085107,...
            -0.06737255472372559380457322478,    0.38611006682276285041905816429, ...
            0.8127236354494134953442245286,      0.4170051844232390480477852360, ...
            -0.07648859907828075427761388352,    -0.05943441864643108730683975853, ...
            0.023680171946847768805923800248,    0.005611434819368834245631844722, ...
            -0.0018232088709110320946081317474,  -0.0007205494455203469950740392992  ];
    end
    
	if Par==3,
	f = [	-0.00379351286438080167548512785,     0.00778259642567274575655572678, ...
            0.0234526961420771662427503659,       -0.0657719112814693671835016013, ...
            -0.0611233900029725412769257018,      0.405176902409118199272476181, ...
            0.793777222626087174791808049,        0.428483476377369981014779057, ...
            -0.071799821619154834013236764,       -0.0823019271062998184866387774, ...
            0.0345550275732977330127285765,       0.0158805448636694509418667488, ...
            -0.00900797613673062389869059132,     -0.00257451768813679701027860332,... 
            0.001117518770830630223506747086,     0.000466216959820402869469087739, ...
            -0.0000709833025063790056111913832,   -0.0000345997731972727738834567238    ];	
	end
	if Par==4,
	f = [	0.0008923139025370029644343566865134121737383, ...
            -0.0016294924252267858123213544087120913158161, ...
            -0.007346167936268049768871523349607132118705,...
            0.01606894713157502651287763071920848497, ...
            0.02668230466960483260703486399047107194, ...
            -0.08126671024919372334475952281575884847,...
            -0.05607731960356925565970517606545448541,...
            0.4153084270006822731294692964046182739,...
            0.78223893442428258982647576237494028015, ...
            0.43438603311435654244291809279435858752, ...
            -0.06662747236681715660425633300924304564,...
            -0.09622042453595263696014466746929391735,...
            0.03933442260558914633132668418129075104, ...
            0.02508225333794960681821191769918797238, ...
            -0.01521172818769721159723577657069625373, ...
            -0.0056582838001308837068552084504470416875092,...
            0.003751434697146086349179148345354230545201, ...
            0.0012665610789256602060212989509419260626456, ...
            -0.000589020224633216477985278808337632907002, ...
            -0.0002599743371222568031968013258330909296978, ...
            0.0000623388543127871811259366755429554393222, ...
            0.0000312298615991952653049475548892070819932, ...
            -3.259647940030750678302345930717926596838865*10^-6, ...
            -1.78499091449334668126754831733930940802101*10^-6	];
	end
	if Par==5,
    f = [   -0.0002120818620674939996481902944136410414047, ...
            0.0003585777411617576912682212568176855692809, ...
            0.0021782943778456947603953983474818156341871, ...
            -0.00415931262757863965550073295937883585, ...
            -0.01013158484690027491468212287890872587, ...
            0.02340832211892778307799159801168871895, ...
            0.02816974427053235189367516327102039599, ...
            -0.09192158806008608329572694314371482072, ...
            -0.05204667025355475665111819740782967482, ...
            0.42157126673075435177306894142751870946, ...
            0.77429362286032745160292654060366328807, ...
            0.43798230665916331792682237681050656764, ...
            -0.06203775157498195089253405688027513479, ...
            -0.10556315130733722646960604243367891954, ...
            0.04128753047211783146902068483243193175, ...
            0.03267479946705735095365614221278235227, ...
            -0.01975839160096546513890474068990811885, ...
            -0.00915950733867616299494402122283339758, ...
            0.00676152022062041680244768896723500979, ...
            0.00243157544253828849057946143095553203, ...
            -0.00166162730392987877456474532860529682, ...
            -0.0006375589261258811091711893522661460267423, ...
            0.0003018579416682447498632500676391103210754, ...
            0.0001403563281237324269902880098227066632525, ...
            -0.0000412198619242655021970132036154651202458, ...
            -0.0000212702216725156138191910478670808515531, ...
            3.7007277113394795164497963176580918909*10^-6, ...
            2.061220398578878156703380849619661817616506*10^-6, ...
            -1.623799517204833517470973891374972010523*10^-7, ...
            -9.6040101127678921250276359452964329236689*10^-8     ];
	end  
    
    if Par==6,
    f = [   0.0000507754878363405645519850534163840843165, ...
            -0.0000811700262678483995046181490233613985991, ...
            -0.0006246130439256835305179694992775427, ...
            0.00109162471232590294442838718434499367, ...
            0.00353901987154099797676760943833152463, ...
            -0.0070294063910027282793426873312339407, ...
            -0.0122315777900379124123273915243790564, ...
            0.0296457728913238384799204963927926239, ...
            0.02878611434666556772409353364797199855, ...
            -0.09967300204601174242984746022593529319, ...
            -0.04876407217567387113947461609898074377, ...
            0.42581954501283846862581982539011507381, ...
            0.76840325757989240980175151406668436853, ...
            0.44040119112685278573807906593541182503, ...
            -0.05810891797261479980792242839864804553, ...
            -0.11226080796481722835521943382584647799, ...
            0.04185249067613626961123692570448481718, ...
            0.03888132625151075695394346028782838898, ...
            -0.02295015327984906593564514384952096935, ...
            -0.01265006790873235128443679361740086299, ...
            0.0095910901759040523779620252393710208, ...
            0.0038576582705936865436976506707985877, ...
            -0.00307393950720855902712037718563727489, ...
            -0.00115743501342733471307799963492014289, ...
            0.00076985473075072663977499623866755922, ...
            0.00032522235901024078543874829743760233, ...
            -0.0001545771992797995031124549459146267672004, ...
            -0.0000752800430693596467873933651612964301517, ...
            0.0000247365593287232279603704234893930555683, ...
            0.0000131398513540214409493534116753473623518, ...
            -2.9243855597575228935454275531139308156*10^-6, ...
            -1.6596192951024207899178040891830917997*10^-6, ...
            2.2559978528161819589808604991860492120455*10^-7, ...
            1.35032449935614466786292988820562494224512*10^-7, ...
            -8.48714339626243656886371228346090530144*10^-9, ...
            -5.309088417196893107804939769143263956288*10^-9     ];
    end

    if Par==7,
    f = [   -0.0000122222506240657722515671975289535857529, ...
            0.0000187113550014121788669954534985164043626, ...
            0.00017510216778483177090611162179579673, ...
            -0.00028720237535706120447042471464670833, ...
            -0.00116931442857976331460374091300655394, ...
            0.00210577204141054775881535169988563035, ...
            0.00482944656070203826696178626298526452, ...
            -0.00993889526908057960572882007938886914, ...
            -0.01380255423628839963367753293728221376, ...
            0.03491050510474272385481698744097843335, ...
            0.028937041983523145219795493887430926, ...
            -0.105556168221561286439976038553264, ...
            -0.046033397038466299456031132666155, ...
            0.428888807249422575091962148625108, ...
            0.763815365416733324438841273059202, ...
            0.442137461401842576513194010237757, ...
            -0.054751241648150457251886153404776, ...
            -0.11729357104319278932816154494452, ...
            0.041705357602576791714703721102373, ...
            0.043993046163079415501169643274255, ...
            -0.025154257568539024269946527565349, ...
            -0.01594684681956793914268132891215748334, ...
            0.01205233824184162260808679717618820937, ...
            0.00543131644288009511292299727400037979, ...
            -0.00461784213043311846868060141284295354, ...
            -0.00180153728333304248757700982439011759, ...
            0.00143474185665241227759908616270387381, ...
            0.00057949944823409528234503848853150457, ...
            -0.00036906682873489535803766645994529048, ...
            -0.00016781721215484972023899538482096261, ...
            0.00007971050025993866031131537354202557,...
            0.00004043048241714020221822226691074551, ...
            -0.0000142356369784515012815941734288435861855, ...
            -7.7712435473118614835496973116889804922*10^-6, ...
            2.0020780498554181311877559696846812794*10^-6, ...
            1.1579769069489572558429154565596032359*10^-6, ...
            -2.069320524393852523963248525127217416*10^-7, ...
            -1.2550913190794570507641433432157*10^-7, ...
            1.3935103885216451572958098811601798785366*10^-8, ...
            8.796593384856986494818762397984045253733*10^-9, ...
            -4.57833406779295070172598474467844779766*10^-10,...
            -2.990566231736865818869246704034628*10^-10   ];
    end
    
 if Par==8,
    f = [   2.9543365214148866341203791795460856498*10^-6, ...
            -4.3682648203200749764069604318588208253*10^-6, ...
            -0.00004829631521409294057518294825489293, ...
            0.0000754736783816503960229241373762259,...
            0.00037129499560741243768356918203220247, ...
            -0.000623560447457940255198422634539581, ...
            -0.0017832600085971970719359953695042521, ...
            0.00330082501061611042248776517113013568, ...
            0.00599484919215588549270790383484069611, ...
            -0.012742370632719795227140215081485, ...
            -0.014978462081708433651627145924743, ...
            0.039372037877979845479596171012811, ...
            0.0288286217592880052319436606959, ...
            -0.110169976983470152889118086934849, ...
            -0.043718983365945585858147809196713, ...
            0.431209815555087578822264464482984, ...
            0.760113302017940493745162524428594, ...
            0.443442549841526025692112583885702, ...
            -0.051860743161188676553989081315791, ...
            -0.121211168231496469284998600458647, ...
            0.041185806676256539334644934268549, ...
            0.048252371085682255135905588083649, ...
            -0.026656710542648603662636753258155,...
            -0.018985244695254866708424157255454, ...
            0.014117470077618781611698797024736, ...
            0.007065827011035095989298841611896, ...
            -0.006156659548258420573069188973804, ...
            -0.00254400371024527345236589153040776921, ...
            0.00223564942204810318055673180460398378, ...
            0.00089677606307967974957481016931868077, ...
            -0.00068717164334800449242695373185778567, ...
            -0.00029777893219563999563276515831158818, ...
            0.00018169287648431022077202247294296659, ...
            0.00008754452091843061802876156718818633, ...
            -0.00004147478606916181410858912360731484, ...
            -0.0000218020007670103510682515219710687, ...
            8.03150299544078598193257904121228*10^-6, ...
            4.4969364435793914273785796464354612539*10^-6, ...
            -1.2754542996407563978284360515425311001*10^-6, ...
            -7.515021558886325576948331510944306958*10^-7, ...
            1.589351722153064827424614721183710204*10^-7, ...
            9.77241850836779841772791655334636543*10^-8, ...
            -1.45400085337535265756124266441*10^-8, ...
            -9.27120559154629669404888252582*10^-9, ...
            8.66999508233871095247749588290123221829*10^-10, ...
            5.70481033390973565849135525930314738442*10^-10, ...
            -2.52542349388545683771492079914226*10^-11, ...
            -1.70798959470554833493549816182562*10^-11  ];
    end
    if Par==9,
    f = [   -7.16492043124788565371403735975293033*10^-7, ...
            1.0293200668945786770405655113884578169*10^-6, ...
            0.000013158885645425329074979104138, ...
            -0.00001978720443246248037040643941679284, ...
            -0.00011443395278590282934418618091927519, ...
            0.00018228485966342260297101418742753238, ...
            0.00062647303213971596119107011420463634, ...
            -0.00107545827274123797178950374028066576, ...
            -0.002421241673651648413361568634198, ...
            0.004597056424920538374827274829998, ...
            0.007022340460196237076042911551157, ...
            -0.015376649629718763882304372940094, ...
            -0.015860223894792907821940082437559, ...
            0.043181727608250448896169058852939, ...
            0.028572667556949285468700689166121, ...
            -0.11388350819004508033470728829253, ...
            -0.04172611020585279692877750305357, ...
            0.433026751103154194437395990290928, ...
            0.757045523384378918975393007517394, ...
            0.444457893176447943769221898235793, ...
            -0.049348866293629167579705877288589, ...
            -0.124345589539290622348246574254961, ...
            0.040473767455728960140784447715682, ...
            0.051844615686247315779705446014627, ...
            -0.027661239498680462969772984800219, ...
            -0.021754553510948844842224556518026, ...
            0.015818715815925058996552193553275, ...
            0.00870275744622918259271751762546, ...
            -0.007614042448258917034725577898924, ...
            -0.00335767452658657858715596317976, ...
            0.003113227788384303748628662551515, ...
            0.001269690925135339902432735868585, ...
            -0.001095745627952606905597323576061, ...
            -0.000462729085505304259448585243503, ...
            0.00033691386928482709390080467507365391, ...
            0.00015541352126673885121571732427354289, ...
            -0.0000913559550874647589464752293334731, ...
            -0.00004613708198462493130010596723498088, ...
            0.00002177639641002902622301367776630784, ...
            0.00001181440945157869423274009466836094, ...
            -4.48811147515276409894428352856613*10^-6, ...
            -2.5723835744866872047576035075266140512*10^-6, ...
            7.802480329370884277020245409489075814*10^-7, ...
            4.67958476945429845632280519885662523*10^-7, ...
            -1.109667018087942287625120969747463337*10^-7, ...
            -6.91654704121803749849331616866102242*10^-8, ...
            1.23752566198101243190950477343359362*10^-8, ...
            7.9740058868468296718526928424325241*10^-9, ...
            -1.0136275688170465755191268077623618*10^-9, ...
            -6.723464414885983563896598608983936*10^-10, ...
            5.417100964283037831066085959289093026*10^-11, ...
            3.6861797364451787327421599467079620161*10^-11, ...
            -1.416273550918584069801563786746191819*10^-12, ...
            -9.85843726123707759474964697066431047*10^-13    ];
    end
end

if strncmpi(Type,'Daubechies',3),
    if Par==2,
        f = [ 0.707106781186547524400844362104849, 0.707106781186547524400844362104849 ];
    end
	if Par==4,  
		f = [	0.4829629131445341433748716, 0.836516303737807905575294, ...
                0.224143868042013381025973, -0.1294095225512603811744494 	];
	end
	if Par==6, 
		f = [	0.332670552950082615998512,  0.806891509311092576494494, ...
                0.459877502118491570095152, -0.135011020010254588696390, ...
               -0.085441273882026661692819,  0.0352262918857095366027407 	];
	end
	if Par==8,
		f = [ 	0.2303778133088965,     0.7148465705529158, ...
                0.630880767929859,     -0.02798376941686011, ...
               -0.1870348117190932,     0.0308413818355608, ...
                0.03288301166688522,   -0.01059740178506904 	];
	end
	if Par==10,
		f = [	0.1601023979741924,     0.6038292697971881, ...
                0.7243085284377715,     0.1384281459013217, ...
               -0.2422948870663802,    -0.03224486958463778, ...
                0.07757149384004565,   -0.006241490212798174, ...
               -0.01258075199908194,    0.003335725285473757 	];
	end
	if Par==12,
		f = [	0.11154074335011,       0.4946238903984554, ...
                0.7511339080210982,     0.315250351709197, ...
               -0.2262646939654429,    -0.1297668675672638,...
                0.0975016055873231,     0.02752286553030565, ...
               -0.03158203931748625,    0.0005538422011615105, ...
                0.004777257510945544,  -0.001077301085308486 	];
	end
	if Par==14,
		f = [	0.07785205408500813,    0.3965393194819123, ...
                0.7291320908462274,     0.4697822874051917, ...
               -0.1439060039285563,    -0.2240361849938672, ...
                0.07130921926683042,    0.080612609151082, ...
               -0.03802993693501439,   -0.016574541630667, ...
                0.01255099855609955,    0.0004295779729213739, ...
               -0.001801640704047446,   0.0003537137999745171 		];
	end
	if Par==16,
		f = [	0.05441584224310704,    0.3128715909143165,  ...
                0.6756307362973218,     0.5853546836542239, ...
               -0.01582910525637238,   -0.2840155429615815, ...
                0.0004724845739030209,  0.1287474266204823, ...
               -0.01736930100181088,   -0.04408825393079791, ...
                0.01398102791739956,    0.00874609404740648, ...
               -0.004870352993451852,  -0.000391740373376942, ...
                0.0006754494064506183, -0.0001174767841247786 		];
	end
	if Par==18,
		f = [	0.03807794736388813,        0.2438346746126514, ...
                0.6048231236902548,         0.6572880780514298, ...
                0.1331973858249681,        -0.2932737832793372, ...
               -0.0968407832230689,         0.148540749338104, ...
                0.03072568147931585,       -0.06763282906135907, ...
                0.0002509471148277948,      0.02236166212368439, ...
               -0.004723204757752752,      -0.004281503682464633, ...
                0.001847646883056686,       0.0002303857635232296, ...
               -0.0002519631889427889,      0.00003934732031628112 	];
	end
	if Par==20,
		f = [	0.02667005790054869,        0.188176800077648, ...
                0.527201188931628,          0.6884590394535462, ...
                0.2811723436606982,        -0.2498464243271048, ...
               -0.1959462743773243,         0.127369340335694, ...
                0.0930573646035142,        -0.07139414716638016, ...
               -0.0294575368218849,         0.03321267405931551, ...
                0.003606553566951515,      -0.0107331754833277, ...
                0.001395351747051327,       0.001992405295184184, ...
               -0.0006858566949593225,     -0.0001164668551292262, ...
                0.0000935886703200315,     -0.00001326420289451403 		];
	end
    if Par==22,
       f = [    0.01869429776144806,        0.1440670211504498, ...
                0.4498997643555165,         0.6856867749154562, ...
                0.4119643689476272,        -0.1622752450269621, ...
               -0.2742308468172826,         0.06604358819685894, ...
                0.1498120124663909,        -0.04647995511648684, ...
               -0.06643878569486228,        0.03133509021904213, ...
                0.02084090436017028,       -0.01536482090617611, ...
               -0.003340858873009247,       0.0049284176560525, ...
               -0.0003085928588149355,     -0.00089302325066525, ...
                0.0002491525235524301,      0.00005443907469928305, ...
               -0.00003463498418694142,     0.000004494274277230458   ];
    end
        if Par==24,
       f = [    0.01311225795736534,        0.1095662728222715, ...
                0.3773551352176745,         0.657198722584349, ...
                0.5158864784293156,        -0.04476388565908393, ...
               -0.3161784537592869,        -0.02377925725693821, ...
                0.1824786059298069,         0.00535956967427179, ...
               -0.0964321200976865,         0.0108491302560784, ...
                0.04154627749559747,       -0.01221864906995923, ...
               -0.01284082519846823,        0.00671149900888981, ...
                0.002248607241020708,      -0.002179503618657147, ...
                0.000006545128213682533,    0.0003886530628261407, ...
               -0.0000885041092094801,     -0.00002424154575734139, ...
                0.00001277695221955214,    -0.000001529071758089919   ];
    end
        if Par==26,
       f = [    0.00920213353936357,        0.082861243876398, ...
                0.3119963221728867,         0.6110558511805082, ...
                0.5888895704451372,         0.0869857261666496, ...
               -0.314972907739053,         -0.124576730762086, ...
                0.1794760794355785,         0.07294893365742099, ...
               -0.1058076181950538,        -0.02648840647689916, ...
                0.05613947710301562,        0.002379972253836755, ...
               -0.02383142071161908,        0.003923941449079961, ...
                0.007255589402002825,      -0.002761911234808676, ...
               -0.001315673911943637,       0.000932326130928484, ...
                0.00004925152513188404,    -0.0001651289885636495, ...
                0.00003067853758174376,     0.00001044193057207212, ...
               -0.000004700416479607929 ,   0.0000005220035098765021    ];
    end
        if Par==28,
       f = [    0.006461153459818989,       0.0623647588469322, ... 
                0.2548502677833766,         0.5543056179241174, ...
                0.6311878490950694,         0.2186706877760189, ...
                -0.2716885522429336,        -0.2180335299738394, ...
                0.138395213856541,          0.1399890165735457, ...
                -0.0867484115685856,        -0.07154895550625034, ...
                0.05523712625188016,        0.02698140830446938, ...
                -0.0301853515397028,        -0.005615049530747707, ...
                0.01278949326524909,        -0.000746218989436958, ...
                -0.003849638867994312,      0.001061691085418039, ...
                0.0007080211541344865,      -0.0003868319473184179, ...
                -0.00004177724577935138,    0.00006875504251988474, ...
                -0.00001033720918460207,    -0.000004389704901652653, ...
                0.000001724994675254821,    -0.000000178713996820958      ];
    end
        if Par==30,
       f = [    0.004538537356680069,       0.0467433948433292, ...
                0.2060238637760462,         0.4926317712332494, ...
                0.6458131398235114,         0.339002535383428, ...
                -0.19320413905893,          -0.2888825960016258,  ...
                0.06528295291444258,        0.1901467139017971,  ...
                -0.03966617641454303,       -0.1111209358626346, ...
                0.03387714389352461,        0.05478055052762776, ...
                -0.0257670072911817,        -0.02081005014572826,  ...
                0.01508391800773139,        0.005101000354434229, ...
                -0.006487734552531616,      -0.0002417564910950625,  ...
                0.001943323977748212,       -0.0003734823537271217,  ...
                -0.0003595652439869339,     0.0001558964896924794,  ...
                0.00002579269911910246,     -0.00002813329623232866,  ...
                0.000003362987176654478,    0.000001811270405641324, ...
                -0.0000006316882317817563,  0.00000006133359905269254     ];
    end

         if Par==32,
       f = [    0.003189220905181802,       0.0349077141074775,  ...
                0.1650642824989111,         0.4303127204089899,  ...
                0.6373563289234388,         0.4402902557886062, ...
                -0.0897510867287953,        -0.3270633068118058,  ...
                -0.02791820715372535,       0.2111906930487478,  ...
                0.02734026408611786,        -0.1323883043443139, ...
                -0.00623972263724492,       0.07592423555847598,  ...
                -0.00758897425298305,       -0.03688839741760147,  ...
                0.01029765955546528,        0.01399376876290007, ...
                -0.006990014507518413,      -0.003644279596729619,  ...
                0.003128023357662664,       0.000407896978913364,  ...
                -0.000941021742187743,      0.000114241519113091,  ...
                0.0001747872440135933,      -0.00006103596571228747,  ...
                -0.00001394566888488284,    0.00001133660857799308, ...
                -0.000001043571333041443,   -0.0000007363656730469882,  ...
                0.0000002308784069376313,   -0.0000000210933961377445    ];
    end
         if Par==34,
       f = [    0.002241806968367765,       0.02598539333038641,  ...
                0.1312149014643511,         0.3703507191428474,  ...
                0.6109966080619875,         0.5183157592365552, ...
                0.02731497388861195,        -0.3283207398752789,  ... 
                -0.1265997478695799,        0.1973105883690036,  ...
                0.1011354893285621,         -0.1268156885448092, ... 
                -0.05709141812622551,       0.081105985705437,  ...
                0.02231233608959475,        -0.04692243752178137,  ...
                -0.003270955473782776,      0.02273367623263168, ...
                -0.003042989911563062,      -0.00860292137975392,  ...
                0.002967996640915282,       0.002301205207197428,  ...
                -0.001436845280352317,      -0.0003281325149411173,  ...
                0.0004394654201169656,      -0.00002561010931458864, ...
                -0.0000820480308801988,     0.00002318681330990614, ...
                0.000006990600842366534,    -0.000004505942411707292,  ...
                0.0000003016549532645506,   0.0000002957700881589635,  ...
                -0.0000000842394830828037,  0.000000007267492843919008     ];
    end
          if Par==36,
       f = [    0.001576310332632241,        0.01928853309434481,  ...
                0.1035884729715391,          0.3146789620466176,  ...
                0.571826841995251,           0.5718016803655575, ...
                0.147223099399332,           -0.2936540837163994,  ...
                -0.2164809618743174,         0.1495339814252923,  ...
                0.1670813196471977,          -0.0923318969776604, ...
                -0.1067522571200224,         0.0648872212223416,  ...
                0.05705125157931265,         -0.04452614611490133,  ...
                -0.02373321210978654,        0.02667070832113655, ...
                0.006262168357742094,        -0.01305148206344844,  ...
                0.0001186301071328846,       0.004943344018360076,  ...
                -0.001118732786346494,       -0.001340596411265555,  ...
                0.0006284657384942994,       0.0002135815764103265,  ...
                -0.000198648570821057,       -0.000000153591634265962, ...
                0.00003741238184339052,      -0.00000852060341054129,  ...
                -0.00000333263477007513,     0.00000176871313748643,  ...
                -0.00000007691633640217469,  -0.0000001176098869880653,  ...
                0.00000003068836137122469,   -0.000000002507934683892356     ];
    end
        if Par==38,
       f = [    0.001108669779715294,         0.01428109865333334,  ...
                0.081278114333354,            0.2643884347822977,  ...
                0.5244363819574067,           0.6017045501513535, ...
                0.2608949440110274,           -0.2280914100170829,  ...
                -0.285838641929714,           0.07465227262054114,  ...
                0.2123497512548378,           -0.03351853842979753, ...
                -0.1427856935054576,          0.02758435493215239,  ...
                0.0869067594236619,           -0.02650123589611068,  ...
                -0.04567422669495623,         0.02162376812192859, ...
                0.01937555029280247,          -0.01398838901012597,  ...
                -0.00586692239134182,         0.007040747519198927,  ...
                0.0007689543646753964,        -0.002687551858597481, ...
                0.0003418086639330359,        0.0007358025360798398,  ...
                -0.0002606761416764582,       -0.0001246007941078683,  ...
                0.0000871127066319985,        0.000005105950548947162,  ...
                -0.00001664017665533139,      0.000003010964385934741,  ...
                0.000001531931507655374,      -0.0000006862755810090276, ...
                0.00000001447088339408005,    0.00000004636937873589416,  ...
                -0.000000011164020912898,     0.000000000866684902796269    ];
    end
        if Par==40,
       f = [    0.0007799530020084384,        0.0105493864101072,  ...
                0.06342373157542249,          0.2199419467839922,  ...
                0.4726958375631425,           0.6104928215175741, ...
                0.3615021297395791,           -0.139211825416023,  ...
                -0.3267863905078842,          -0.01672694530514085,  ...
                0.2282909876975237,           0.03985032729018178, ...
                -0.1554585361790331,          -0.02471674917392653,  ...
                0.1022916746204368,           0.005632268726873665,  ...
                -0.06172283526148656,         0.005874682288534986, ...
                0.03229427583633914,          -0.00878931595226129,  ...
                -0.01381051445886118,         0.006721621652169426,  ...
                0.004420538864131319,         -0.003581491222634283, ...
                -0.00083156152944895,         0.001392558453825609,  ...
                -0.00005349753868856166,      -0.0003851044297986765,  ...
                0.0001015328014373285,        0.00006774275277093538,  ...
                -0.00003710583043522718,      -0.000004376140493506968,  ...
                0.000007241242222701708,      -0.000001011993125412585, ...
                -0.0000006847073928591012,    0.0000002633921999175421,  ...
                0.0000000002014328820034285,  -0.0000000181484172957345, ...
                0.000000004056123630675098,   -0.0000000002998833944499773     ];
    end
        if Par==42,
       f = [    0.0005488240399453808,        0.007776660464348811,  ...
                0.04924790475876491,          0.1813601028599902,  ...
                0.419688998145241,            0.6015074510688103, ...
                0.4445910837993439,           -0.03572381948901234,  ...
                -0.33566575122537,            -0.1123978514710653,  ...
                0.2115648260162405,           0.1152333439473735, ...
                -0.1399410472763452,          -0.08177625782428998,  ... 
                0.09660066710664022,          0.04572352417673011,  ...
                -0.06497770623152748,         -0.01865389796875268, ...
                0.03972696757220106,          0.003357765554657301,  ...
                -0.02089211624987374,         0.002403482102825579,  ...
                0.008988852342563074,         -0.002891344156898007, ...
                -0.002958382842307337,        0.001716612683276365,  ...
                0.0006394203289590759,        -0.0006906733219030776,  ...
                -0.00003196410553726866,      0.0001936652571660039,  ...
                -0.0000363553295677002,       -0.00003499676704742804,  ...
                0.00001535487521020741,       2.79033850314008*10^-6, ...
                -3.090027001911602*10^-6,     3.16610662424439*10^-7,  ...
                2.99214595113828*10^-7,       -1.000404119487493*10^-7,  ...
                -2.254019869522092*10^-9,     7.058055911572644*10^-9,  ...
                -1.471958939283684*10^-9,     1.038808947669207*10^-10    ];
    end
        if Par==44,
       f = [    0.0003862673246197253,          0.005721914066631049,  ...
                0.03807032187691932,            0.1483689789282081,  ...
                0.3677320057234413,             0.5784372354311235, ...
                0.5079033273631367,             0.07372115020105462,  ...
                -0.3127333476121842,            -0.2005720141344328,  ...
                0.1640948426591233,             0.179974931810035, ...
                -0.0971123372197599,            -0.1317696149504392,  ...
                0.06807740848784511,            0.08455839833964807,  ...
                -0.05136497255398131,           -0.04653131832736136, ...
                0.03697137276735332,            0.02058693268949487,  ...
                -0.02348031395539096,           -0.006213835918293782,  ...
                0.01256489065516637,            0.0003001305020824184, ...
                -0.005455761185358356,          0.001044278408986017,  ...
                0.001827032986409597,           -0.000770702101944467, ...
                -0.0004237923063271874,         0.0003286138886837352,  ...
                0.0000434593692542139,          -0.00009405347080647135,  ...
                0.00001137454223403893,         0.00001737397675279249, ...
                -6.166816318076451*10^-6,       -1.565197277819435*10^-6,  ...
                1.295199441207159*10^-6,        -8.78003044824892*10^-8,  ...
                -1.283352833826032*10^-7,       3.761280659022215*10^-8,  ...
                1.680187679020641*10^-9,        -2.729659356918619*10^-9,  ...
                5.33601149622179*10^-10,        -3.60216327759258*10^-11   ]; 
    end
        if Par==46,
       f = [    0.0002719278182602901,          0.004203109552950134,  ...
                0.02931247643736339,            0.1205254471036576,  ...
                0.3184759568589838,             0.5449708209347766, ...
                0.5510501337055957,             0.1813841378320262,  ...
                -0.2614398761995617,            -0.2714429864972958,  ...
                0.0921245749243952,             0.2235864349031235, ...
                -0.03304774793732929,           -0.164030308293076,  ...
                0.02028436820991752,            0.1123069840244809,  ...
                -0.0211292480280753,            -0.07021415427385447, ...
                0.02176834655240395,            0.03849895908078205,  ...
                -0.01852549112315692,           -0.01753870400805271,  ...
                0.01275326613768589,            0.006032371159860696, ...
                -0.00707603267773538,           -0.001134947880346942,  ...
                0.003123184807392083,           -0.000246537026777104,  ...
                -0.001061334361043996,          0.000319454992361999,  ...
                0.0002567865998070605,          -0.0001500373078747796,  ...
                -0.00003379188332733358,        0.00004426515179248939, ...
                -2.635561787093299*10^-6,       -8.348692795439998*10^-6,  ...
                2.397822036092728*10^-6,        8.148343038224153*10^-7,  ...
                -5.339546450998099*10^-7,       1.853340229309923*10^-8,  ...
                5.418084825798256*10^-8,        -1.400079829615052*10^-8,  ...
                -9.473736128438874*10^-10,      1.050551729533758*10^-9, ...
                -1.93260193304542*10^-10,       1.250331739337031*10^-11    ];
    end
        if Par==48,
       f = [    0.0001914240079776934,          0.003081894336144903,  ...
                0.02248099723913652,            0.09725657409395711,  ...
                0.272893661713225,              0.5043448957614517, ...
                0.5749146829767083,             0.2809851510053765, ... 
                -0.1872418464658568,            -0.3179111538203686,  ...
                0.004781510762825361,           0.2392258659829295, ...
                0.042531243536347,              -0.1711600617797226,  ...
                -0.03877318682438014,           0.1210092088290207,  ...
                0.02098022912439134,            -0.08215538086453539, ...
                -0.004578395730450242,          0.05129798128535279,  ...
                -0.004944235600686442,          -0.02821125709939177,  ...
                0.007661004281903052,           0.01304905186620713, ...
                -0.006290964935213451,          -0.004746267936383896,  ...
                0.00373576397589871,            0.001153694353296646,  ...
                -0.001696334910033699,          -0.00004416435334971148,  ...
                0.0005860851561798487,          -0.000118113728929818,  ...
                -0.0001459980983446589,         0.00006558881863639525, ...
                0.00002183096348720674,         -0.00002022741617379432,  ...
                1.337052417608915*10^-8,        3.900821594914755*10^-6,  ...
                -8.979550384702172*10^-7,       -4.032228084773544*10^-7,  ...
                2.166180932866001*10^-7,        -5.054643465620961*10^-10,  ...
                -2.255577015054618*10^-8,       5.157391468496204*10^-9, ...
                4.748066278754132*10^-10,       -4.024365393060184*10^-10,  ...
                6.991284124010881*10^-11,       -4.342457865150871*10^-12    ];
    end
        if Par==50,
       f = [    0.0001349140099190347,          0.002258811421928896,  ...
                0.01720076826024011,            0.07809905498823061,  ...
                0.2318787664012299,             0.4600423810287615, ...
                0.5820659400767751,             0.3680932821030804,  ...
                -0.0973914656330337,            -0.3369053502250008,  ...
                -0.08780821918435377,           0.2245876085890194, ... 
                0.118127995249711,              -0.1507901382415503,  ...
                -0.09866413500669811,           0.1066759749914026,  ...
                0.06677659879664328,            -0.07716829308409851, ...
                -0.03721412413495515,           0.05365935822485196,  ...
                0.01555326324900022,            -0.03407222558276132,  ...
                -0.003082422156835764,          0.01893883135517686, ...
                -0.001991310900912701,          -0.00886823166517666,  ... 
                0.002729324227909382,           0.003325513663082694,  ...
                -0.001844076742849583,          -0.0009007311640167205,  ...
                0.0008780108981336599,          0.0001154141401669138,  ...
                -0.0003101446827543741,         0.00003546902687895924, ...
                0.00007911359754985811,         -0.00002735413947239519,  ...
                -0.00001278271963847119,        8.998380584459695*10^-6,  ...
                5.236962340729562*10^-7,        -1.780721007596221*10^-6,  ...
                3.214832263002398*10^-7,        1.924439055598693*10^-7,  ...
                -8.66437988387495*10^-8,        -2.613650299345624*10^-9, ...
                9.287154489811888*10^-9,        -1.882036245600092*10^-9,  ...
                -2.230357347052905*10^-10,      1.537216399287396*10^-10,  ...
                -2.529795306330366*10^-11,      1.510990558374748*10^-12     ] ;
    end
end

if strncmpi(Type,'Symmlet',3),
    if Par==1,
        f = [ 0.707106781186547524400844362104849, 0.707106781186547524400844362104849 ];
    end
	if Par==2,  
		f = [	0.4829629131445341433748716, 0.836516303737807905575294, ...
                0.224143868042013381025973, -0.1294095225512603811744494 	];
	end
	if Par==3, 
		f = [	0.332670552950082615998512,  0.806891509311092576494494, ...
                0.459877502118491570095152, -0.135011020010254588696390, ...
               -0.085441273882026661692819,  0.0352262918857095366027407 	];    
    end
	if Par==4,
		f = [   -0.07576571478950221,       -0.029635527646002493,   ...
                0.497618667632775,          0.8037387518051321,   ...
                0.29785779560530606,        -0.09921954357663353,   ...
                -0.012603967262031304,      0.032223100604051466	];
	end
	if Par==5,
		f = [   0.0195388827352498268,      -0.0211018340246890410,   ...
                -0.175328089908056224,      0.016602105764510848,   ...
                0.63397896345679206,        0.72340769040404079,   ...
                0.199397533976855597,       -0.039134249302313844,   ...
                0.0295194909257062613,      0.0273330683449987688		];
	end
	if Par==6,  
		f = [   0.01540410932704474,        0.003490712084221531,   ...
                -0.1179901111485212,        -0.04831174258569789,   ...
                0.4910559419279768,         0.7876411410286536,  ...
                0.3379294217281644,         -0.07263752278637825,   ...
                -0.02106029251237119,       0.04472490177078142,   ...
                0.001767711864253766,       -0.007800708325032496	];
	end
	if Par==7,
		f = [   0.01026817670846495,        0.004010244871523197,   ...
                -0.1078082377032895,        -0.1400472404429405,   ...
                0.2886296317506303,         0.7677643170048699,  ...
                0.5361019170905749,         0.01744125508685128,   ...
                -0.04955283493703385,       0.06789269350122353,   ...
                0.03051551316588014,        -0.01263630340323927,  ...
                -0.001047384888679668,      0.002681814568260057 				];
	end
	if Par==8, 
		f = [	 -0.003382415951003908,     -0.000542132331797018,   ...
                 0.03169508781151886,       0.00760748732494897,   ...
                 -0.1432942383512576,       -0.06127335906765891,   ...
                 0.4813596512592537,        0.7771857516996492,   ...
                 0.3644418948360139,        -0.05194583810802026,   ...
                 -0.02721902991713553,      0.04913717967372511,   ...
                 0.003808752013880463,      -0.01495225833706814,   ...
                 -0.0003029205147226741,    0.001889950332768561 		];
	end
	if Par==9,
		f = [	 0.001069490032908175,      -0.0004731544986808867,   ...
                 -0.01026406402762793,      0.008859267493410117,   ...
                 0.06207778930285313,       -0.01823377077946773,   ...
                 -0.1915508312971598,       0.03527248803579076,   ...
                 0.6173384491414731,        0.7178970827644066,   ...
                 0.2387609146068536,        -0.05456895843120489,   ...
                 0.0005834627459892242,     0.03022487885821281,   ...
                 -0.01152821020772933,      -0.01327196778183437,   ...
                 0.0006197808889867399,     0.001400915525915921	];
	end
	if Par==10,
		f = [	 0.0007701598091036597,     0.00009563267068491565,   ...
                 -0.008641299277002591,     -0.001465382581138532,   ...
                 0.04592723923095083,       0.0116098939028464,   ...
                 -0.1594942788849671,       -0.0708805357805798,   ...
                 0.4716906669415791,        0.7695100370206782,   ...
                 0.3838267610640166,        -0.03553674047551473,   ...
                 -0.03199005688220715,      0.04999497207760673,   ...
                 0.005764912033412411,      -0.02035493981234203,   ...
                 -0.0008043589319389408,    0.004593173585320195,   ...
                 0.00005703608359777954,    -0.0004593294210107238 			];
	end
    if Par==11,
        f = [    0.0004892636102790465,     0.00011053509770077,   ...
                 -0.006389603666537886,     -0.002003471900538333,   ...
                 0.04300019068196203,       0.03526675956730489,   ...
                 -0.1446023437042145,       -0.2046547945050104,   ...
                 0.2376899090326669,        0.7303435490812422,   ...
                 0.5720229780188006,        0.09719839447055164,   ...
                 -0.02283265101793916,      0.06997679961196318,   ...
                 0.03703741598066749,       -0.0240808415947161,   ...
                 -0.009857934828835874,     0.006512495674629366,   ...
                 0.0005883527354548924,     -0.001734366267274675,   ...
                 -0.00003879565575380471,   0.0001717219506928879    ];
	end
    if Par==12,
        f = [    -0.0001790665869786187,     -0.0000181580788773471,   ...
                 0.002350297614165271,       0.0003076477963025531,   ...
                 -0.01458983644921009,       -0.002604391031185636,   ...
                 0.05780417944546282,        0.01530174062149447,   ...
                 -0.1703706972388913,        -0.07833262231005749,   ...
                 0.4627410312313846,         0.7634790977904264,   ...
                 0.3988859723844853,         -0.0221623061807925,   ...
                 -0.03584883074255768,       0.04917931829833128,   ...
                 0.007553780610861577,       -0.02422072267559388,   ...
                 -0.001408909244210085,      0.007414965517868044,   ...
                 0.0001802140900854918,      -0.001349755755614803,  ...
                 -0.00001135392805049379,    0.0001119671942470856     ];
	end
    if Par==13,
        f = [    0.00007042986709788876,     0.00003690537416474083,   ...
                 -0.0007213643852104347,     0.0004132611973679777,   ...
                 0.00567485376954048,        -0.00149244724795732,   ...
                 -0.02074968632748119,       0.01761829684571489,   ...
                 0.09292603099190611,        0.008819757923922775,  ...
                 -0.1404900930989444,        0.1102302225796636,   ...
                 0.6445643835707201,         0.6957391508420829,   ...
                 0.1977048192269691,         -0.1243624606980946,   ...
                 -0.05975062792828035,       0.01386249731469475,   ...
                 -0.01721164274779766,       -0.02021676815629033,   ...
                 0.005296359721916584,       0.007526225395916087,   ...
                 -0.0001709428497111897,     -0.001136063437095249,   ...
                 -0.0000357386241733562,     0.00006820325245288671     ];
	end
    if Par==14,
        f = [    0.00004461898110644152,     0.00001932902684197359,   ...
                 -0.0006057602055992672,     -0.00007321430367811753,  ...
                 0.004532677588409982,       0.001013142476182283,  ...
                 -0.01943931472230284,       -0.002365051066227485,  ...
                 0.06982761641982026,        0.02589859164319225,  ...
                 -0.1599974161449017,        -0.05811184934484923,  ...
                 0.4753357348650867,         0.7599762436030552,  ...
                 0.3932015487235067,         -0.03531809075139569,  ...
                 -0.05763449302747868,       0.03743308903888159,  ...
                 0.004280522331795536,       -0.02919621738508546,  ...
                 -0.002753775776578359,      0.01003769335863697,   ...
                 0.000366476770515625,       -0.002579441672422145,   ...
                 -0.00006286548683867455,    0.0003984356519092697,   ...
                 0.00001121086996816579,     -0.00002587908845615303     ];
	end
    if Par==15,
        f = [    0.000028660707399511618,    0.00002171788180828788,   ...
                 -0.00040216855725720875,    -0.00010815440716833184,   ...
                 0.0034810287335767247,      0.0015261379816940614,   ...
                 -0.01717125379613366,       -0.008744790172337504,   ...
                 0.06796982852319061,        0.06839330814883558,   ...
                 -0.13405630898495416,       -0.19662638205515043,   ...
                 0.24396267904975416,        0.7218430182315695,   ...
                 0.5786404260768322,         0.11153371778633057,   ...
                 -0.04108264654576916,       0.040735492211698726,   ...
                 0.021937649685129356,       -0.03887671343231794,   ...
                 -0.019405010502865788,      0.010079976851620261,   ...
                 0.0034234504500168204,      -0.003590165561744339,   ...
                 -0.0002673165193090753,     0.0010705671659500386,   ...
                 0.00005512253152945822,     -0.0001606618644005603,   ...
                 -7.359664544811822*10^-6,   9.712420111966456*10^-6   ];
	end
    if Par==16,
        f = [    -0.000010797985444060825,   -5.3964960594326874*10^-6,   ...
                 0.0001654568107575125,      0.00003656600318814429,   ...
                 -0.001338720814110487,      -0.00022211728990382076,   ...
                 0.006937761277326152,       0.0013598467013441263,   ...
                 -0.024952761177797576,      -0.00351028784332523,   ...
                 0.07803784806236971,        0.030721150497025267,   ...
                 -0.15959221677565338,       -0.05404070849544753,   ...
                 0.4753426930640303,         0.7565249925494576,   ...
                 0.3971230417468278,         -0.034574136859023705,   ...
                 -0.06698302069147749,       0.03233309920319836,   ...
                 0.004869285298401547,       -0.031051198943356957,   ...
                 -0.003126520186702648,      0.012666730821216037,   ...
                 0.0007182134482533401,      -0.0038809118096674354,   ...
                 -0.00010844597239036113,    0.0008523546022172409,   ...
                 0.000028078644213177975,    -0.00010943145921454754,   ...
                 -3.1135618941800743*10^-6,  6.230004737262791*10^-6   ];
	end
    if Par==17,
        f = [    4.2973506629986956*10^-6,   2.7801688647505625*10^-6,   ...
                 -0.00006293696990392763,    -0.000013506396803548528,   ...
                 0.00047599681001429773,     -0.00013864025450273018,   ...
                 -0.002741673289197601,      0.0008567721508596517,   ...
                 0.010482379074328828,       -0.004819175075474626,   ...
                 -0.03329133163029172,       0.017904007873400175,   ...
                 0.10475473990614559,        0.017271442186243685,   ...
                 -0.1185666208590123,        0.1423985355331832,   ...
                 0.6507166482523769,         0.6814888952815437,   ...
                 0.18053935219589717,        -0.15507633780169236,   ...
                 -0.0860711490606599,        0.016158692741874165,   ...
                 -0.0072616521550352585,     -0.01803889816207702,   ...
                 0.009952983267225308,       0.012396993805630462,   ...
                 -0.0019054056198378236,     -0.003932327920107898,   ...
                 0.00005839844001157818,     0.0007198270212964605,   ...
                 0.000025208218893204052,    -0.00007607121225156968,   ...
                 -2.452745060439947*10^-6,   3.791246551426962*10^-6   ];
	end
    if Par==18,
        f = [    2.6140187028148005*10^-6,   1.3597256411120972*10^-6,   ...
                 -0.000045260099151022276,   -0.000014073709856766986,   ...
                 0.0003962647703053168,      0.00007059203923948904,   ...
                 -0.002314325309127317,      -0.00041350784379068383,   ...
                 0.009503146264507371,       0.001648937806891975,   ...
                 -0.03033018760379904,       -0.00510180660253373,   ...
                 0.08421406918773965,        0.034024019305781264,   ...
                 -0.15998146275864744,       -0.05223243824458295,   ...
                 0.47375599169059157,        0.7536405256423095,   ...
                 0.4016913034776937,         -0.0323059809889792,   ...
                 -0.07374895242999176,       0.028536771134301076,   ...
                 0.006292693377765397,       -0.03170873328122627,   ...
                 -0.0032688666608867664,     0.01500986633652236,   ...
                 0.0010909833425781458,      -0.005238814002156449,   ...
                 -0.0001897773119942463,     0.0014277813050762765,   ...
                 0.000047650786275655254,    -0.0002657505546476236,   ...
                 -9.890224232299607*10^-6,   0.00002954545735292322,   ...
                 7.866685781899692*10^-7,    -1.5123391551795506*10^-6   ];
	end
    if Par==19,
        f = [    1.6822101374310987*10^-6,   1.5736233900352247*10^-6,   ...
                 -0.000029084130082422266,   -0.000016290552278010628,   ...
                 0.00027537426211342433,     0.00010455921733678424,   ...
                 -0.0017655593234062173,     -0.0006524693932405551,   ...
                 0.008240792309156801,       0.003903523253914178,   ...
                 -0.028979337236055096,      -0.01858176253516738,   ...
                 0.08265001836815583,        0.09052455442917737,   ...
                 -0.12553103109717628,       -0.1933024666109857,   ...
                 0.24104659627671224,        0.7125038682183499,   ...
                 0.5857258835982894,         0.12559052557160855,   ...
                 -0.051417687498444706,      0.019505347664693754,   ...
                 0.012438665386547568,       -0.04432391883451796,   ...
                 -0.022155886952494982,      0.015464312015532047,   ...
                 0.0076454965207025565,      -0.005198489304476271,   ...
                 -0.0011616402526739716,     0.0021058625600138578,   ...
                 0.00015660905060254328,     -0.0006245052918345595,   ...
                 -0.00004106344742558738,    0.0001139524500214609,   ...
                 7.487464336562946*10^-6,    -0.000011966487988902881,   ...
                 -5.34322686486931*10^-7,    5.711932378486931*10^-7    ];
	end
    if Par==20,
        f = [    -6.299929076393514*10^-7,   -1.788381654723433*10^-7,   ...
                 0.000013267792274307285,    5.715455163913879*10^-6,   ...
                 -0.00012638480147829343,    -0.00006060770430172846,   ...
                 0.0007316472152793887,      0.00026095442734033774,   ...
                 -0.003170803376848161,      -0.00043732670748901353,   ...
                 0.012090589576786489,       0.0019211676342072246,   ...
                 -0.03663262584755033,       -0.013703513480057003,   ...
                 0.07424185717192522,        0.021714054253910882,    ...
                 -0.1609037041111548,        -0.03440166000144136,  ... 
                 0.4894006264128903,         0.7507502982136451,   ... 
                 0.3886194419202543,         -0.04521583665884769,   ...
                 -0.07854049760146452,       0.038482757401527116,   ...
                 0.021702142828213884,       -0.023834246491111964,   ...
                 -0.0005786000109801912,     0.01717848841075394,   ...
                 0.0006324619136888966,      -0.007342682291732647,   ...
                 -0.0005183408270565881,     0.002204695518355125,   ...
                 0.00017752923953843423,     -0.00048232152518013383,   ...
                 -0.00003434395791274426,    0.00007406549050117618,   ...
                 3.2530365938639585*10^-6,   -7.413187876923192*10^-6,   ...
                 -1.0539262586695314*10^-7,  3.7126642825501105*10^-7   ];
	end
    if Par==21,
        f = [    2.4666687655533695*10^-7,   7.914248389737127*10^-8,   ...
                 -5.098478772975259*10^-6,   -1.8332525199414118*10^-6,  ... 
                 0.00004694842985874322,     6.088937039539296*10^-6,   ...
                 -0.00028648787462784116,    0.0000620228593209885,   ...
                 0.0014213616519049683,      -0.000432897763752902,   ...
                 -0.005353988694277162,      0.0017655806658982696,   ...
                 0.015402169297438343,       -0.008988082355110115,   ...  
                 -0.04716066845359608,       0.010671178775815944,   ...
                 0.10149319625091788,        0.007388554132196435,   ...
                 -0.12355469595545877,       0.14411131060308363,   ...
                 0.6461733765248382,         0.6815923286559533,   ...
                 0.19203957652810966,        -0.1513192612796282,   ...
                 -0.08711940361369748,       0.02790507569203459,   ...
                 0.00625184607285568,        -0.015083074728218489,   ...
                 0.011187152328141188,       0.01450803878862172,   ...
                 -0.004094954089873776,      -0.006749289429682182,   ...
                 0.0006955275549391147,      0.002031175814309811,   ...
                 -0.00002051693327729468,    -0.00041348833873143716,   ...
                 -0.000010572800078769339,   0.000058204303827933644,   ...
                 1.8409309631189882*10^-6,   -5.161165596805668*10^-6,   ...
                 -7.41575536253381*10^-8,    2.3113011977605634*10^-7   ];
	end
    if Par==22,
        f = [    1.5469892096783629*10^-7,   5.5743916934275646*10^-8,   ...
                 -3.4968422679833534*10^-6,  -1.4370363121605532*10^-6,   ...
                 0.000037380396807580654,    0.00001758094055422716,   ...
                 -0.00024483364534030704,    -0.00010585417623386842,   ...
                 0.0011466583097184356,      0.0003226285920630637,   ...
                 -0.004398948577001492,      -0.0005350221246800008,   ...
                 0.015051009425518232,       0.003114197433178795,   ...
                 -0.03983484859283483,       -0.014316309998756563,   ...
                 0.07823387878992308,        0.023785426002814004,   ...
                 -0.1593358890186262,        -0.029991258060954408,   ...
                 0.4905406567562972,         0.7486689893282149,   ...
                 0.38946084846091966,        -0.04646770047156702,   ...
                 -0.08505017544453836,       0.03554141495048075,   ...
                 0.023467779250349607,       -0.02426659585361798,   ...
                 -0.0019440502814989976,     0.01780655171188534,   ...
                 0.0003877388938834633,      -0.008837354866518892,   ...
                 -0.0006023488806178174,     0.0030309320519889987,   ...
                 0.00025194457519336455,     -0.0007945748785723063,   ...
                 -0.00006608462189094136,    0.00015493987521617144,   ...
                 0.000010202851313391099,    -0.000021749524061348802,   ...
                 -8.277216867302219*10^-7,   2.011485781107503*10^-6,   ...
                 3.24106151598103*10^-8,     -8.994488134257862*10^-8    ];
	end
    if Par==23,
        f = [    -6.088466993156109*10^-8,   -1.6615204304595472*10^-8,   ...
                 1.4054134729420521*10^-6,   5.065545930484757*10^-7,   ...
                 -0.000014639159377583806,   -3.935849604669885*10^-6,   ...
                 0.00009610455472740267,     -3.235179654211356*10^-7,   ...
                 -0.0004912392092331126,     0.0001158628187825532,    ...
                 0.0020794420582160053,      -0.0005397239075479521,   ...
                 -0.006723311075642093,      0.0023801024676674445,   ...
                 0.018427107312302106,       -0.009708672972735352,   ...
                 -0.052966594269723166,      0.006024328570134602,   ...
                 0.09868192677853799,        0.0026702039747034177,   ...
                 -0.1299766448268073,        0.13550672099058436,   ...
                 0.6387382588927035,         0.6863848550710899,   ...
                 0.20649267004039606,        -0.14369715652684245,   ...
                 -0.08651297991591521,       0.032763741913238865,   ...
                 0.012533684345771838,       -0.014070855072454728,   ...
                 0.01062794083550241,        0.015024638946099386,   ...
                 -0.004866076235991217,      -0.007967029103383086,   ...
                 0.0010722016128121928,      0.0028093755854430584,   ...
                 -0.00008313779295501525,    -0.0006970208534264938,   ...
                 -0.00001326126189449906,    0.00012603730932265127,   ...
                 4.446291783441227*10^-6,    -0.000016139269667555554,   ...
                 -4.77553521844591*10^-7,    1.3365100765666046*10^-6,   ...
                 1.523804807305629*10^-8,    -5.5838351515000746*10^-8   ];
	end
    if Par==24,
        f = [    -5.2761865436863985*10^-8,  -8.690427954503197*10^-8,   ...
                 1.0423208461508862*10^-6,   1.3969259517138812*10^-6,   ...
                 -0.000011118982675005505,   -0.000012657478427953705,   ...
                 0.0000773048286026619,      0.00007342863308434417,   ...
                 -0.00039834902286264043,    -0.0003016488762115482,   ...
                 0.001633599109508149,       0.0009177057961552579,   ...
                 -0.005702626293286814,      -0.002525604130488947,   ...
                 0.017026080554935406,       0.007861303957162372,   ...
                 -0.040982600612933254,      -0.021283796275817276,   ...
                 0.08372034358295327,        0.05437314304210483,   ...
                 -0.12818178218050752,       -0.035646434246813574,   ...
                 0.46244307077589863,        0.7501914811592109,   ...
                 0.4182481983661332,         -0.04590755369598393,   ...
                 -0.1204529195912711,        0.010585497932626633,   ...
                 0.023513400494410315,       -0.022539152872931466,   ...
                 -0.0050215879773052394,     0.01833813846077915,   ...
                 0.0021823244607757,         -0.009954852329009839,   ...
                 -0.0015175250406422927,     0.0038357471744951996,   ...
                 0.0007028294509160195,      -0.0011116911888874527,   ...
                 -0.00021118156525038934,    0.0002507142923455971,   ...
                 0.0000442170717380261,      -0.000043237491892894634,   ...
                 -6.45525681719476*10^-6,    5.343044886829498*10^-6,   ...
                 5.953891366605391*10^-7,    -4.1947862803617735*10^-7,   ...
                 -2.5954357139211734*10^-8,  1.5757584826821534*10^-8   ];
	end
    if Par==25,
        f = [   1.3559364540611227*10^-8,    -9.795737094326415*10^-10,   ...
                 -3.5324698866183274*10^-7,  -2.864726752719305*10^-8,   ...
                 4.222032788604233*10^-6,    5.49693105547489*10^-7,   ...
                 -0.00003092830508017534,    -4.212835064605567*10^-7,   ...
                 0.00016479057910264333,     -0.000037508744144871775,   ...
                 -0.0007330652888770748,     0.0002658102686466399,   ...
                 0.002714164795025192,       -0.000968322030306876,   ...
                 -0.00785635517951613,       0.0037969122058442175,   ...
                 0.02059795248498291,        -0.012308050256422173,   ...
                 -0.05640015507726106,       0.007187328235056668,   ...
                 0.0956457447512976,         -0.0058027884901894556,   ...
                 -0.12231754226725715,       0.1631514710968921,   ...
                 0.6543719294101015,         0.6715904122541644,   ...
                 0.18132650835646258,        -0.15599473760485902,   ...
                 -0.08362502581802364,       0.041304200470714036,   ...
                 0.017768444024218653,       -0.013548122888083406,   ...
                 0.010296151957318115,       0.014633468106540926,   ...
                 -0.006440853043345421,      -0.008929653602321046,   ...
                 0.0019327781450455884,      0.0036421681089588136,   ...
                 -0.00033935744667514006,    -0.0010791779886949856,   ...
                 0.000027173893640607642,    0.000238698525771045,   ...
                 1.0500229544977402*10^-6,   -0.000039832648663228424,   ...
                 -5.410954871718931*10^-7,   4.768328003833541*10^-6,   ...
                 3.284183959667723*10^-8,    -3.759486167421689*10^-7,   ...
                 1.085292734525874*10^-9,    1.5021122420804858*10^-8    ];
	end
    
end
	
if strncmpi(Type,'Vaidyanathan',3),
	f = [	-.000062906118	.000343631905	-.000453956620	...
			-.000944897136	.002843834547	.000708137504	...
			-.008839103409	.003153847056	.019687215010	...
			-.014853448005	-.035470398607	.038742619293	...
			.055892523691	-.077709750902	-.083928884366	...
			.131971661417	.135084227129	-.194450471766	...
			-.263494802488	.201612161775	.635601059872	...
			.572797793211	.250184129505	.045799334111		];
end

if strncmpi(Type,'Battle',3),
	if Par == 1,
     g = [0.578163    0.280931    -0.0488618   -0.0367309 ...
          0.012003    0.00706442  -0.00274588  -0.00155701 ...
          0.000652922 0.000361781 -0.000158601 -0.0000867523
	    ];
	end

	if Par == 3,
        
	g = [0.541736    0.30683      -0.035498    -0.0778079 ...
         0.0226846   0.0297468    -0.0121455   -0.0127154 ...
         0.00614143  0.00579932   -0.00307863  -0.00274529 ...
         0.00154624  0.00133086   -0.000780468 -0.00065562 ...
	     0.000395946 0.000326749  -0.000201818 -0.000164264 ...
         0.000103307
	    ];
	end

	if Par == 5,
 	g = [0.528374    0.312869    -0.0261771   -0.0914068 ...
         0.0208414   0.0433544   -0.0148537   -0.0229951  ...
         0.00990635  0.0128754   -0.00639886  -0.00746848 ...
         0.00407882  0.00444002  -0.00258816  -0.00268646 ...
         0.00164132  0.00164659  -0.00104207  -0.00101912 ...
	     0.000662836 0.000635563 -0.000422485 -0.000398759 ...
	     0.000269842 0.000251419 -0.000172685 -0.000159168 ...
	     0.000110709 0.000101113
	    ];
	end
        l = length(g);
        f = zeros(1,2*l-1);
        f(l:2*l-1) = g;
        f(1:l-1) = reverse(g(2:l));
end
if strncmpi(Type,'Lemarie',3),
     f = [  3.0620193e-005  3.8631712e-005 -5.9878645e-005 ...
-7.4997553e-005 1.1762624e-004 1.4604017e-004 ...
-2.3234459e-004 -2.8538040e-004 4.6210986e-004  ...
 5.5993014e-004 -9.2718796e-004 -1.1037270e-003 ...
 1.8821116e-003 2.1866870e-003 -3.8823801e-003 ...
-4.3537889e-003  8.2013819e-003  8.6851939e-003 ...
-1.7982112e-002 -1.7176116e-002 4.2067863e-002 ...
 3.2080525e-002 -1.1003574e-001 -5.0201141e-002 ...
 4.3391759e-001 7.6612115e-001 4.3391759e-001...
 -5.0201141e-002 -1.1003574e-001  3.2080525e-002  ...
4.2067863e-002 -1.7176116e-002 -1.7982112e-002 ...
8.6851939e-003 8.2013819e-003 -4.3537889e-003 ...
-3.8823801e-003 2.1866870e-003 1.8821116e-003 ...
-1.1037270e-003 -9.2718796e-004  5.5993014e-004 ...
 4.6210986e-004  -2.8538040e-004 -2.3234459e-004  ...
1.4604017e-004  1.1762624e-004 -7.4997553e-005 ...
-5.9878645e-005 ];
end
if strncmpi(Type,'Pollen',3),
     s = 2 * sqrt(2);
     if length(Par) == 1
         phi = Par(1);
 f =[ (1 + cos(phi) - sin(phi))/s ...
      (1 + cos(phi) + sin(phi))/s ...
      (1 - cos(phi) + sin(phi))/s ...
      (1 - cos(phi) - sin(phi))/s];
     end
     if length(Par)==2
    phi1 = Par(1);
    phi2 = Par(2);
    s = 2 * sqrt(2);
f=[( 1+cos(phi1) - cos( phi2 ) - cos( phi1) * cos( phi2) ...
       + sin( phi1 ) - cos( phi2) * sin( phi1) - sin( phi2 )  ...
       + cos( phi1) * sin( phi2) - sin( phi1) * sin( phi2 ) )/ (2*s) ...
   ( 1-cos( phi1) + cos( phi2) - cos( phi1) * cos( phi2) ...
       + sin( phi1) + cos( phi2) * sin( phi1) - sin( phi2) ...
       - cos( phi1) * sin( phi2) - sin( phi1) * sin( phi2) )/ (2*s) ...
   ( 1 + cos( phi1) * cos( phi2) + cos( phi2) * sin( phi1) ...
       - cos( phi1) * sin( phi2 ) + sin( phi1) *sin( phi2 ) )/s ...
   ( 1 + cos( phi1 ) * cos( phi2 ) - cos( phi2 ) * sin( phi1 ) ...
       + cos( phi1 )* sin( phi2 ) + sin( phi1)*  sin( phi2) )/s ... 
   ( 1-cos( phi1) + cos( phi2 )- cos( phi1 )* cos( phi2 ) ...
       - sin( phi1 ) - cos( phi2 ) * sin( phi1 ) + sin( phi2 ) ...
       + cos( phi1 )* sin( phi2 )- sin( phi1 )* sin( phi2) ) /(2*s)...
   ( 1+cos( phi1 )- cos( phi2 )- cos( phi1 )* cos( phi2 ) ...
       - sin( phi1 ) + cos( phi2 ) * sin( phi1 ) + sin( phi2 ) ...
       - cos( phi1 ) * sin( phi2 ) - sin( phi1 )*  sin( phi2) )/ (2*s)];
     end
end
%f = f ./ norm(f);

% 
% Copyright (c) 1993-5. Jonathan Buckheit and David Donoho
%     
% Extension by Daphne Lai and Brani Vidakovic
%     
    
   
    
%   
% Part of WaveLab Version 802
% Built Sunday, October 3, 1999 8:52:27 AM
% This is Copyrighted Material
% For Copying permissions see COPYING.m
% Comments? e-mail wavelab@stat.stanford.edu
%   
    
